// @HEADER
// *****************************************************************************
//   Zoltan2: A package of combinatorial algorithms for scientific computing
//
// Copyright 2012 NTESS and the Zoltan2 contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#include <Zoltan2_OrderingProblem.hpp>
#include <Zoltan2_XpetraCrsMatrixAdapter.hpp>
#include <Zoltan2_TestHelpers.hpp>
#include <iostream>
#include <fstream>
#include <limits>
#include <vector>
#include <Teuchos_ParameterList.hpp>
#include <Teuchos_RCP.hpp>
#include <Teuchos_CommandLineProcessor.hpp>
#include <Tpetra_CrsMatrix.hpp>
#include <Tpetra_Vector.hpp>
#include <MatrixMarket_Tpetra.hpp>

using Teuchos::RCP;

/////////////////////////////////////////////////////////////////////////////
// Program to demonstrate use of Zoltan2 to order a TPetra matrix
// (read from a MatrixMarket file or generated by Galeri::Xpetra).
// Usage:
//     a.out [--inputFile=filename] [--outputFile=outfile] [--verbose]
//           [--x=#] [--y=#] [--z=#] [--matrix={Laplace1D,Laplace2D,Laplace3D}
// Karen Devine, 2011
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
// Eventually want to use Teuchos unit tests to vary z2TestLO and
// GO.  For now, we set them at compile time based on whether Tpetra
// is built with explicit instantiation on.  (in Zoltan2_TestHelpers.hpp)

typedef zlno_t z2TestLO;
typedef zgno_t z2TestGO;
typedef zscalar_t z2TestScalar;

typedef Tpetra::CrsMatrix<z2TestScalar, z2TestLO, z2TestGO> SparseMatrix;
typedef Tpetra::Vector<z2TestScalar, z2TestLO, z2TestGO> Vector;
typedef Vector::node_type Node;
typedef Zoltan2::XpetraCrsMatrixAdapter<SparseMatrix> SparseMatrixAdapter;

size_t computeBandwidth(RCP<SparseMatrix> A, z2TestLO *iperm)
// Returns the bandwidth of the (local) permuted matrix
// Uses the inverse permutation calculated from the OrderingSolution
// if passed in, otherwise is calculating the original value.
{
  z2TestLO ii, i, j, k;
  typename SparseMatrix::local_inds_host_view_type  indices;
  typename SparseMatrix::values_host_view_type values;

  z2TestLO bw_left = 0;
  z2TestLO bw_right = 0;

  z2TestLO  n = A->getLocalNumRows();

  // Loop over rows of matrix
  for (ii=0; ii<n; ii++) {
    A->getLocalRowView (ii, indices, values);
    for (k=0; k< static_cast<z2TestLO>(indices.size()); k++){
      if (indices[k] < n){ // locally owned
        if (iperm){
          i = iperm[ii];
          j = iperm[indices[k]];
        } else {
          i = ii;
          j = indices[k];
        }
        if (j-i > bw_right)
          bw_right = j-i;
        if (i-j > bw_left)
          bw_left = i-j;
      }
    }
  }

  // Total bandwidth is the sum of left and right + 1
  return (bw_left + bw_right + 1);
}

/////////////////////////////////////////////////////////////////////////////
int main(int narg, char** arg)
{
  std::string inputFile = "";            // Matrix Market file to read
  std::string outputFile = "";           // Output file to write
  bool verbose = false;                  // Verbosity of output
  int testReturn = 0;
  size_t origBandwidth = 0;
  size_t permutedBandwidth = 0;

  ////// Establish session.
  Tpetra::ScopeGuard tscope(&narg, &arg);
  Teuchos::RCP<const Teuchos::Comm<int> > comm = Tpetra::getDefaultComm();

  int me = comm->getRank();

  // Read run-time options.
  Teuchos::CommandLineProcessor cmdp (false, false);
  cmdp.setOption("inputFile", &inputFile,
                 "Name of a Matrix Market file in the data directory; "
                 "if not specified, a matrix will be generated by Galeri.");
  cmdp.setOption("outputFile", &outputFile,
                 "Name of file to write the permutation");
  cmdp.setOption("verbose", "quiet", &verbose,
                 "Print messages and results.");
  if (me == 0) std::cout << "Starting everything" << std::endl;

  //////////////////////////////////
  // Even with cmdp option "true", I get errors for having these
  //   arguments on the command line.  (On redsky build)
  // KDDKDD Should just be warnings, right?  Code should still work with these
  // KDDKDD params in the create-a-matrix file.  Better to have them where
  // KDDKDD they are used.
  int xdim=10;
  int ydim=10;
  int zdim=10;
  std::string matrixType("Laplace3D");

  cmdp.setOption("x", &xdim,
                "number of gridpoints in X dimension for "
                "mesh used to generate matrix.");
  cmdp.setOption("y", &ydim,
                "number of gridpoints in Y dimension for "
                "mesh used to generate matrix.");
  cmdp.setOption("z", &zdim,
                "number of gridpoints in Z dimension for "
                "mesh used to generate matrix.");
  cmdp.setOption("matrix", &matrixType,
                "Matrix type: Laplace1D, Laplace2D, or Laplace3D");

  //////////////////////////////////
  // Ordering options to test.
  //////////////////////////////////
  std::string orderMethod("rcm"); // TODO: Allow "RCM" as well
  cmdp.setOption("order_method", &orderMethod,
                "order_method: natural, random, rcm, sorted_degree");

  std::string orderMethodType("local"); // TODO: Allow "LOCAL" as well
  cmdp.setOption("order_method_type", &orderMethodType,
                "local or global or both");

  //////////////////////////////////
  cmdp.parse(narg, arg);


  RCP<UserInputForTests> uinput;

  if (inputFile != ""){ // Input file specified; read a matrix
    uinput = rcp(new UserInputForTests(testDataFilePath, inputFile, comm, true));
  }
  else                  // Let Galeri generate a matrix

    uinput = rcp(new UserInputForTests(xdim, ydim, zdim, matrixType, comm, true, true));

  RCP<SparseMatrix> origMatrix = uinput->getUITpetraCrsMatrix();

  if (me == 0)
    std::cout << "NumRows     = " << origMatrix->getGlobalNumRows() << std::endl
         << "NumNonzeros = " << origMatrix->getGlobalNumEntries() << std::endl
         << "NumProcs = " << comm->getSize() << std::endl;

  ////// Create a vector to use with the matrix.
  // Currently Not Used
  /*
  RCP<Vector> origVector, origProd;
  origProd   = Tpetra::createVector<z2TestScalar,z2TestLO,z2TestGO>(
                                    origMatrix->getRangeMap());
  origVector = Tpetra::createVector<z2TestScalar,z2TestLO,z2TestGO>(
                                    origMatrix->getDomainMap());
  origVector->randomize();
  */

  ////// Specify problem parameters
  Teuchos::ParameterList params;
  params.set("order_method", orderMethod);
  params.set("order_method_type", orderMethodType);

  ////// Create an input adapter for the Tpetra matrix.
  SparseMatrixAdapter adapter(origMatrix);

  ////// Create and solve ordering problem
  try
  {
  Zoltan2::OrderingProblem<SparseMatrixAdapter> problem(&adapter, &params);
  if (me == 0) std::cout << "Going to solve" << std::endl;
  problem.solve();

  ////// Basic metric checking of the ordering solution

  Zoltan2::LocalOrderingSolution<z2TestLO> *soln =
    problem.getLocalOrderingSolution();

  if (me == 0) std::cout << "Going to get results" << std::endl;
  // Check that the solution is really a permutation

  z2TestLO * perm = soln->getPermutationView();

  if (outputFile != "") {
    std::ofstream permFile;

    // Write permutation (0-based) to file
    // each process writes local perm to a separate file
    //std::string fname = outputFile + "." + me;
    std::stringstream fname;
    fname << outputFile << "." << comm->getSize() << "." << me;
    permFile.open(fname.str().c_str());
    size_t checkLength = soln->getPermutationSize();
    for (size_t i=0; i<checkLength; i++){
      permFile << " " << perm[i] << std::endl;
    }
    permFile.close();

  }

  if (me == 0) std::cout << "Verifying results " << std::endl;

  // Verify that checkPerm is a permutation
  testReturn = soln->validatePerm();

  // Compute original bandwidth
  origBandwidth = computeBandwidth(origMatrix, nullptr);

  // Compute permuted bandwidth
  z2TestLO * iperm = soln->getPermutationView(true);
  permutedBandwidth = computeBandwidth(origMatrix, iperm);

  } catch (std::exception &e){
      std::cout << "Exception caught in ordering" << std::endl;
      std::cout << e.what() << std::endl;
      std::cout << "FAIL" << std::endl;
      return 0;
  }

  if (testReturn)
    std::cout << me << ": Not a valid permutation" << std::endl;

  int gTestReturn;
  Teuchos::reduceAll<int,int>(*comm, Teuchos::EReductionType::REDUCE_MAX, 1,
                     &testReturn, &gTestReturn);

  int increasedBandwidth = (permutedBandwidth > origBandwidth);
  if (increasedBandwidth) 
    std::cout << me << ": Bandwidth increased: original " 
              << origBandwidth << " < " << permutedBandwidth << " permuted "
              << std::endl;
  else 
    std::cout << me << ": Bandwidth not increased: original " 
              << origBandwidth << " >= " << permutedBandwidth << " permuted "
              << std::endl;

  int gIncreasedBandwidth;
  Teuchos::reduceAll<int,int>(*comm, Teuchos::EReductionType::REDUCE_MAX, 1,
                     &increasedBandwidth, &gIncreasedBandwidth);

  if (me == 0) {
    if (gTestReturn)
      std::cout << "Solution is not a permutation; FAIL" << std::endl;
    else if (gIncreasedBandwidth && (orderMethod == "rcm"))
      std::cout << "Bandwidth was increased; FAIL" << std::endl;
    else
      std::cout << "PASS" << std::endl;

  }

}

